#!/bin/bash

#DEBUGFILE="/tmp/debugfile"

VERSION="0.1.55"
if [[ "$1" == "" ]] || [[ "$2" == "" ]]; then 
	echo "$0: must supply mode and route"
	exit 1
fi

AST='*'
VTYSH_PROG=`which vtysh`
VTYSH_PAGER=more
LOGFACPRI=local5.info
LOGSTART=1
RTBH_DB="/var/local/rtbh/rtbh.db"
SQLITE_INIT="/var/local/rtbh/sqlite.rc"
SQLITE_PROG=`which sqlite3`
SQL_RETRY=20
db_mode=$1
DUMP_DELIM=" , "
DEFAULT_EXPIRY="60 MINUTES"
#LONGEST_ACTIVE must correspond to the cron job that deletes all routes
LONGEST_ACTIVE="16 DAYS"
RECIDIVE_TIMES=10
RECIDIVE_WIDE_TIMES=24
RECIDIVE_FORCEACT="60 MINUTES"
RECIDIVE_ACTIVE="90 MINUTES"
LONGEST_KEEP="2 MONTH"
TABLE_CREATE="CREATE TABLE rtbh(seq  INTEGER PRIMARY KEY AUTOINCREMENT, route,mask,lowner,rowner,fullstring,stamp default CURRENT_TIMESTAMP,times default 0,active default 0,added default 0,deleted default 0,expires default (datetime('now','+60 MINUTES')), geoip_cc VARCHAR(2) ); CREATE TABLE rtbh_status(stamp default CURRENT_TIMESTAMP);"
SQL_STAMP=" stamp = CURRENT_TIMESTAMP "

GEOIPCONF="/var/local/rtbh/geoip.dat"
GEOIPSTRING="geoipblock"
#GEOIPNAMES="ssh router honey cygwin firewall lists"
GEOIPNOTNAMES="dns dns-query $GEOIPSTRING"
GEOIPDEFEXPIRY="3 HOURS"
GEOIPLOOKUP="geoiplookup"
DO_COPROC=${DO_COPROC:-"1"}

argv=($*)
argc=${#argv[@]}

if [[ "$DEBUGFILE" != "" ]]; then echo "$0 $*" >> $DEBUGFILE; fi


for method in USER id LOGNAME; do
	case "$method" in

		USER)
			;;
		id)
			USER="`id -un`"
			;;
		LOGNAME)
			USER=${LOGNAME}
			;;
	esac
	if [[ "$USER" != "" ]]; then break; fi
done

shift


rtbh_geoip_enabled()
{
	if ! [[ -r "$GEOIPCONF" ]]; then return 1; fi
	if ! [[ -x "$GEOIPLOOKUP" ]]; then
		if [[ "`which $GEOIPLOOKUP 2>/dev/null`" == "" ]]; then
			return 1
		fi
	fi
	return 0
}

netmask_to_bits () {
   c=0 x=0$( printf '%o' ${1//./ } )
   while [ $x -gt 0 ]; do
       let c+=$((x%2)) 'x>>=1'
   done
   echo $c ;
}

v4dec() {
        for i; do
                echo $i | {
                        local IFS=./
                        read a b c d e
                        test -z "$e" && e=32
                        echo -n "$((a<<24|b<<16|c<<8|d)) $((-1<<(32-e))) "
                }
        done
}

log(){ local x=$1 n=2 l=-1;if [ "$2" != "" ];then n=$x;x=$2;fi;while((x));do let l+=1 x/=n;done;echo $l; }

v4decpipe() {
	(while read IN; do echo "`v4dec $IN`"; done)
}

inet_ntoa () {

    local IFS=. num quad ip e slen
    num=$1
    for e in 3 2 1
    do
        (( quad = 256 ** e))
        (( ip[3-e] = num / quad ))
        (( num = num % quad ))
    done
    ip[3]=$num
	if [[ "$2" != "" ]]; then
		slen=`log $2`
		if [[ "$slen" != "" ]]; then 
			((slen=32-slen))
			slen="/$slen"
			echo "${ip[*]}${slen}"
			return 0
		fi
	fi
     echo "${ip[*]}"

}

inet_ntoapipe() {
	(while read IP LEN; do inet_ntoa $IP $LEN; done)
}

v4dectest() {
	local IFS="${IFS}/"
        read addr1 mask1 addr2 mask2
        if (( (addr1&mask2) == (addr2&mask2) && mask1 >= mask2 )); then
                return 0
        else
                return 1
        fi
}


rtbh_set_net_mask()
{
	echo $1 | grep -q '/' 2>&1 >/dev/null && SUBNET="/`echo $1 | cut -f2 -d'/'`"
	if [[ "$SUBNET" == "" ]]; then
		NETWORK=$1
		if [[ "$2" != "" ]]; then
			SUBNETBITS="`netmask_to_bits $2 2>/dev/null`"
			SUBNET=" mask $2"
			((LOGSTART++))
		else
			SUBNET="/32"
			SUBNETBITS="32"
		fi
	else
		NETWORK="`echo $1 | cut -f1 -d'/'`"
		SUBNETMASK=`echo $SUBNET | grep -oE "[0-9]+\.[0-9]+\.[0-9]+\[0-9]+"` && SUBNET="/`netmask_to_bits $SUBNETMASK`"
		SUBNETBITS="${SUBNET:1}"
	fi
	return 0
}

rtbh_set_net_mask $*

shift $LOGSTART

if [[ "$SSH_CLIENT" != "" ]]; then
	USERFROM="@`echo $SSH_CLIENT | cut -f1 -d' '`"
else
	USERFROM="@`hostname`"
fi


for ((i=LOGSTART+1 , j=0;i<argc;i++ , j++)); do

	FS[$j]=`printf %q "${argv[$i]}"`
done 

for ((i=LOGSTART+1;i<argc;i++)); do
	echo "${argv[$i]}" | grep -E -q ".+@.+" || continue
	ROWNER="${argv[$i]}"
	break
done


function parse_time_arg ()
{
	if [[ "$1" == "" ]]; then return 1; fi
	OLDIFS="$IFS"
	IFS="${IFS}:/,;\ "
	TIMEARG=($*)
	IFS="$OLDIFS"

	TIME=`printf %d ${TIMEARG[0]}`

	if [[ "${TIME}" == "" ]]; then return 1; fi

	TIMEUNIT="SECONDS"
	if [[ "${TIMEARG[1]}" != "" ]]; then
	case "${TIMEARG[1],,}" in
		
		ye* | yr )
			TIMEUNIT="YEARS"
			;;
		mo* | mt* )
			TIMEUNIT="MONTHS"	
			;;
		we* | w )
			TIMEUNIT="DAYS"
			((TIME *= 7))
			;;
		da* | dy* | d )
			TIMEUNIT="DAYS"
			;;
		ho* | hr* | h )
			TIMEUNIT="HOURS"
			;;
		mi* | mn* | m )
			TIMEUNIT="MINUTES"
			;;
		se* | s )
			TIMEUNIT="SECONDS"
			;;
		*)
			echo "invalid time unit ${TIMEARG[1]}"
			return 1
		;;
	esac
	fi
	echo "$TIME $TIMEUNIT"
	return 0
}

for ((i=LOGSTART+1;i<argc;i++)); do
	case "${argv[$i],,}" in
		$GEOIPSTRING)
			true ${FS_WHERE_NAME:="${argv[$i]}"}
			DEFAULT_EXPIRY="30 MINUTES"
			MAX_EXPIRY="2 DAYS"	
			;;
		router)
			true ${FS_WHERE_NAME:="${argv[$i]}"}
			[[ "$MASTER_EXPIRY" != "Y" ]] && DEFAULT_EXPIRY="45 MINUTES"
			;;
		nt | windows-security)
			true ${FS_WHERE_NAME:="${argv[$i]}"}
			DEFAULT_EXPIRY="45 MINUTES"
			;;
		fail2ban)
			MASTER_EXPIRY="Y"
			DEFAULT_EXPIRY="25 MINUTES"
			MAX_EXPIRY="7 DAYS"
			true ${FS_COULD_WHERE_NAME:="${argv[$i]}"}
			;;
		list | lists)	
			true ${FS_COULD_WHERE_NAME:="${argv[$i]}"}
			DEFAULT_EXPIRY="8 HOURS"
			MAX_EXPIRY="7 DAYS"
			;;
		dshield)
			FS_WHERE_NAME="${argv[$i]}"
			DEFAULT_EXPIRY="4 HOURS"
			;;
		spamhaus)
			FS_WHERE_NAME="${argv[$i]}"
			DEFAULT_EXPIRY="28 HOURS"
			;;
		ssh)
			true ${FS_WHERE_NAME:="${argv[$i]}"}
			[[ "$MASTER_EXPIRY" != "Y" ]] && DEFAULT_EXPIRY="45 MINUTES"
			;;
		vnc)
			true ${FS_WHERE_NAME:="${argv[$i]}"}
			[[ "$MASTER_EXPIRY" != "Y" ]] && DEFAULT_EXPIRY="60 MINUTES"
			;;
		dns-query)
			true ${FS_WHERE_NAME:="${argv[$i]}"}
			[[ "$MASTER_EXPIRY" != "Y" ]] && DEFAULT_EXPIRY="60 MINUTES"
			;;
		asterisk)
			true ${FS_WHERE_NAME:="${argv[$i]}"}
			DEFAULT_EXPIRY="90 MINUTES"
			;;
		exp*)
			if [[ "$i" -lt "$argc" ]]; then
				if (( i+1 < argc )); then
					DEFAULT_EXPIRY=`parse_time_arg ${argv[$i+1]}:${argv[$i+2]}`
					if [[ "$?" -eq "0" ]]; then break; fi
				fi 
				DEFAULT_EXPIRY=`printf "%d" ${argv[$i+1]}`
				if [[ "$DEFAULT_EXPIRY" != "" ]]; then
					DEFAULT_EXPIRY="$DEFAULT_EXPIRY SECONDS"
				else
					DEFAULT_EXPIRY="15 MINUTES"
				fi
			fi
			break
			;;
		maxexp*)
			if [[ "$i" -lt "$argc" ]]; then
				if (( i+1 < argc )); then
					MAX_EXPIRY=`parse_time_arg ${argv[$i+1]}:${argv[$i+2]}`
					if [[ "$?" -eq "0" ]]; then continue; fi
				fi
				unset MAX_EXPIRY
				MAXEXPIRE=`printf %d ${argv[$i+1]}` 
				if [[ "$?" -eq "0" ]]; then continue; fi
			fi
			MAXEXPIRE=${MAXEXPIRE:-"4"}
			;;
	esac	
done

: ${FS_WHERE_NAME:="${FS_COULD_WHERE_NAME:-${FS[0]}}"}

if [[ "$SQLITE_PROG" == "" ]] || [[ ! -x "$SQLITE_PROG" ]]; then
		echo "$0: sqlite program missing or not executable $SQLITE_PROG"
		exit 1
fi
if [[ "$RTBH_DB" == "" ]] || [[ ! -r "$RTBH_DB" ]]; then
		echo "$0: database missing or not usable $RTBH_DB"
		exit 1
fi
SQLITE_EXEC="$SQLITE_PROG -batch -line $RTBH_DB"
if [[ -r "$SQLITE_INIT" ]]; then
	SQLITE_EXEC="$SQLITE_EXEC -init $SQLITE_INIT"
fi

[[ -r "${RTBH_DB}-wal" ]] && dbWALsz=(`du -m "${RTBH_DB}-wal"`) 
[[ ${dbWALsz} != "" ]] && (( dbWALsz > 5 )) && echo "pragma wal_checkpoint(TRUNCATE);" | $SQLITE_EXEC



sqlite_inet_enabled()
{
	OUTPUT=`echo "select \"true\" where (inet_aton(\"192.168.0.0\") = 3232235520);" | $SQLITE_EXEC` 2>/dev/null || return 1
	if [[ "$OUTPUT" != "" ]]; then return 0; fi
	return 1
}


rtbh_expire_time()
{
	if [[ "${AOUTPUT[expires]}" != "" ]] || [[ "$3" != "" ]]; then
		local OUTPUT="min(max(datetime(CURRENT_TIMESTAMP,+ \"${1:-${DEFAULT_EXPIRY}}\"),"
		OUTPUT="$OUTPUT datetime(\"${3:-${AOUTPUT[expires]}}\")), datetime(CURRENT_TIMESTAMP,"
		OUTPUT="$OUTPUT + \"${2:-${LONGEST_ACTIVE:-"8 DAYS"}}\"))"
	else
		OUTPUT="min(datetime(CURRENT_TIMESTAMP,+ \"${1:-${DEFAULT_EXPIRY}}\"),"
		OUTPUT="$OUTPUT datetime(CURRENT_TIMESTAMP, + \"${2:-${LONGEST_ACTIVE:-"8 DAYS"}}\"))"
	fi
	echo "$OUTPUT"
}

rtbh_debug_where()
{
	if [[ "$RTBH_DEBUG_NET" != "" ]]; then
		local OUT="route = \"$RTBH_DEBUG_NET\""
		NETWORK="$RTBH_DEBUG_NET"
		if [[ "$RTBH_DEBUG_MASK" != "" ]]; then
			OUT="$OUT and mask = \"/$RTBH_DEBUG_MASK\""
			SUBNET="$RTBH_DEBUG_MASK"
		fi	
		echo " $OUT and "
	fi
}

#cron or local
if [[ "$LS_COLORS" == "" ]] && [[ "$TERM" == "" ]] && [[ "$SHLVL" == "" ]]; then
	RTBH_CRON_JOB=1
fi
if [[ "$SSH_CLIENT" != "" ]]; then
	RTBH_CRON_JOB=0
	RTBH_SSH_JOB=1
fi

if [[ "$RTBH_CRON_JOB" == "1" ]] && [[ "$RTBH_SSH_JOB" != "1" ]]; then
	(( SQL_RETRY *= 3))
fi

pr_random ()
{
        h=${1:-0}
        l=${2:-10}
        if [[ "${RANDOM}" == "" ]]; then
                n=32768
                while [ $n -ge 32768 ]; do
                        n=1$(</dev/urandom sed 's/[^0-9]*//g' | sed 's/^0//g' | dd bs=5 count=1 2>/dev/null)
                        n=`printf %d $n`
                        if [[ $n -ge 100000 ]]; then
                                n=$((n-100000))
                        fi
                done
        else
                n=$RANDOM
        fi
        echo $(( ( n * ( h-l+1 )) / 32768 + l))

}


read_sql_coproc_reply()
{

	SQL_RET=0
	while read -u "${SQL_IN}" -r LINE; do
		if [[ "$LINE" == "\"EndOfQuery\" = EndOfQuery" ]]; then break; fi #"
		echo $LINE | grep -q "^Error:" && echo $LINE && SQL_RET=1 && return 1 && break
		echo $LINE
	done
	return $SQL_RET
}


sql_exec()
{
	local SQLIN 
	local INPUT
	local SQLOUT
	local SQLRET
	local i=0


	if [[ "$1" != "" ]]; then 
		SQLIN="$*"
		if [[ "${!1}" != "" ]]; then
			SQLIN="${!1}"
		fi
	fi
	if [[ "$SQLIN" == "" ]]; then 
		while read -r INPUT; do
			if [[ "$SQLIN" != "" ]]; then
				SQLIN="${SQLIN}
$INPUT"	
			else
				SQLIN="$INPUT"
			fi
		done
	fi
	if [[ "$SQLIN" == "" ]]; then 
		return 1; 
	fi
	if [[ "$DEBUGFILE" != "" ]]; then echo "$SQLIN" >> $DEBUGFILE; fi
	for ((;i<$SQL_RETRY;i++)); do
		if [[ "$DO_COPROC" == "0" ]]; then 
			SQLOUT="`echo -e "$SQLIN" | $SQLITE_EXEC $SQLITE_ARGS 2>&1`"
		else
			echo -e "$SQLIN" >&"${SQL_OUT}"
			echo "select \"EndOfQuery\";" >&"$SQL_OUT"
			SQLOUT=`read_sql_coproc_reply`
		fi

		SQLRET=$?	
		if ((SQLRET==0)); then
			echo $SQLOUT | grep -q "^Error:.*locked" || echo -e "$SQLOUT" && return 0
		fi
			
		if [[ "$RTBH_CRON_JOB" == "1" ]] && [[ "$RTBH_SSH_JOB" != "1" ]]; then
			sleep `pr_random 1 10`
			continue
		fi
		echo $SQLOUT  >&2
	done
	if [[ "$SQLOUT" != "" ]] && [[ "$SQLRET" == "0" ]] && ((i<SQL_RETRY)); then 
		echo -e "$SQLOUT" 
	elif ((SQLRET>0)); then
		return $SQLRET
	elif ((SQL_RETRY<=i)); then
		return 101
	fi
	return $SQLRET
}

SQLITE_CMD="sql_exec"

if [[ "$DO_COPROC" != "0" ]]; then
	if  [[ -z "$COPROC_PID" ]]; then
		coproc $SQLITE_EXEC $SQLITE_ARGS 2>&1 
		COMMENT="Did coproc $COPROC_PID"
		exec 4<&${COPROC[0]}- 5>&${COPROC[1]}- ; SQL_IN=4 SQL_OUT=5
		echo "pragma optimize;" >&"${SQL_OUT}"
		echo "pragma mmap_size=104857600;" >&"${SQL_OUT}"
		read -u "${SQL_IN}" LINE
	else
		COMMENT="Croproc $COPROC_PID ${COPROC[@]}"
	fi
fi

sqlout_to_aar()
{
	if [[ "$1" == ""  ]] || [[ "$2" == "" ]]; then
		return 1;
	fi
	unset $1
	declare -Ag $1
	local FN
	local FV
	local i=0
	
	if [[ "${!2}" == "" ]]; then
		return 1
	fi
	while read -r TEXT; do
		FN=`echo $TEXT | cut -f1 -d'=' | tr -d '\t '`
		if [[ "$FN" == "" ]]; then continue; fi
		((i++))
		FV=`echo $TEXT | cut -f2 -d'='`
		FV=`printf %q "${FV:1}"`
		eval $1[$FN]="$FV"
	done <<EOF
${!2}
EOF
	((i>0)) && return 0
	return 1
}


rtbh_db_select()
{
	if [[ "$WHERE_TXT" != "" ]]; then WHERE_TXT="$WHERE_TXT and"; fi
	WHERE_TXT="$WHERE_TXT route=\"$NETWORK\" and  mask=\"$SUBNET\""
	case "$db_mode" in
		del | list | refresh)
		;;
		*)
		WHERE_TXT="$WHERE_TXT and lowner=\"$USER$USERFROM\""
		if [[ "$ROWNER" != "" ]]; then WHERE_TXT="$WHERE_TXT and rowner=\"$ROWNER\""; fi
		;;
	esac	
	if [[ "${FS[0]}" != "" ]] && [[ "$FS_WHERE_NAME" != "" ]]; then 
		NOFS_WHERE_TXT="$WHERE_TXT"
		FS_LIKE_TXT="$WHERE_TXT and fullstring like \"% $FS_WHERE_NAME %\""
		if [[ "$db_mode" == "add" ]]; then
			WHERE_TXT="$FS_LIKE_TXT"
			unset FS_LIKE_TXT
		else
			WHERE_TXT="$WHERE_TXT and fullstring=\"${FS[@]}\"" 
		fi
	else
		unset NOFS_WHERE_TXT
		unset FS_LIKE_TXT
	fi
	SELECT_TXT="select * from rtbh where ${WHERE_TXT} $ORDER_TXT $LIMIT_TXT;"
	unset WHERE_TXT
	for ((i=0;i<3;i++)); do
		OUTPUT=`echo "$SELECT_TXT" | $SQLITE_CMD || return $?`
		if [[ "$OUTPUT" != "" ]]; then
			sqlout_to_aar AOUTPUT OUTPUT || return $?
			return 0
		fi
		if [[ "$FS_LIKE_TXT" != "" ]]; then
			SELECT_TXT="select * from rtbh where ${FS_LIKE_TXT} $ORDER_TXT $LIMIT_TXT;"
			unset FS_LIKE_TXT
			continue
		fi
		if [[ "$NOFS_WHERE_TXT" != "" ]]; then 
			SELECT_TXT="select * from rtbh where ${NOFS_WHERE_TXT} $ORDER_TXT $LIMIT_TXT;"
			unset NOFS_WHERE_TXT
			continue
		fi
		break
	done
	return 1
}

rtbh_db_check()
{
	if [[ "$WHERE_TXT" != "" ]]; then WHERE_TXT="$WHERE_TXT and"; fi
	WHERE_TXT="$WHERE_TXT active = 1 and datetime(expires) > datetime(CURRENT_TIMESTAMP)"
	ORDER_TXT="order by 'expires' DESC"
	LIMIT_TXT="LIMIT 1"
	rtbh_db_select || return $?
	if [[ "${AOUTPUT[seq]}" != "" ]]; then
		return 0;
	fi
	return 1
						
}

rtbh_db_list()
{
	FND=0
	local i=0
	SELECT_TXT="select * from rtbh WHERE route=\"$NETWORK\" and  mask=\"$SUBNET\""
	if [[ "$db_mode" == "list" ]]; then
		SELECT_TXT="$SELECT_TXT and active = 1 and datetime(expires) > datetime(CURRENT_TIMESTAMP) "
	fi

	ORDER_TXT="order by 'stamp' DESC"
	while OUTPUT=`echo "$SELECT_TXT $ORDER_TXT LIMIT 1 offset $i;" | $SQLITE_CMD || return $?`; do
		((i++))
		if [[ "$OUTPUT" != "" ]]; then
			sqlout_to_aar AOUTPUT OUTPUT || return $?
		else
			(( FND>0 )) && return 0
			return 1
		fi
		if [[ "${AOUTPUT[seq]}" != "" ]]; then
			(( FND++ ))
			local DB_out=""
			for j in "${!AOUTPUT[@]}"; do
				if [[ "$DB_out" != "" ]]; then 
					DB_out="${DB_out}${DUMP_DELIM}" 
				fi
				DB_out="${DB_out}$j=${AOUTPUT[$j]}"
			done
			echo "$DB_out"
		fi
	done
	(( FND>0 )) && return 0
	return 1
}

rtbh_recidive_wide()
{
	if [[ "${#AOUTPUT[@]}" -lt 4 ]]; then
		rtbh_db_select || return $?
		if [[ "${AOUTPUT[seq]}" == "" ]]; then
			return 1;
		fi
	fi 
	if (( RECIDIVE_WIDE_TIMES < AOUTPUT[times] )); then
		return 1;
	fi

	SELECT_TXT="select sum(times)/sum(max(active,added,deleted)) $AST 2 from rtbh"
	SELECT_TXT="$SELECT_TXT where route = \"${AOUTPUT[route]}\" and mask = \"${AOUTPUT[mask]}\" and datetime(expires) "
	SELECT_TXT="$SELECT_TXT > datetime(CURRENT_TIMESTAMP, \"-${RECIDIVE_WIDE_ACTIVE:-${LONGEST_ACTIVE:-\"8 DAYS\"}}\")"
	SQLOUT=`echo -e "$SELECT_TXT ;" | $SQLITE_CMD` || return $?
	
	SQLOUT=`echo "$SQLOUT" | cut -d= -f2`
	if [[ "$SQLOUT" == "" ]]; then return 1; fi
	if (( SQLOUT >  RECIDIVE_WIDE_TIMES)); then
		echo $SQLOUT
		return 0
	fi
	return 1

}

DIG_CMD="true"
unset DIG_TIMEOUTCMD
which timeout 2>&1 >/dev/null && DIG_TIMEOUT_CMD="timeout -k3 2"
which dig 2>&1 >/dev/null && DIG_CMD="$DIG_TIMEOUT_CMD dig"

drone_bl_expiry()
{

	if [[ "$SUBNET" != "/32" ]]; then (echo 0; return 0); fi

        REVNET=`echo "$NETWORK" | ( IFS='.' read a b c d; echo "$d.$c.$b.$a" )`
	DIG_OUT=`$DIG_CMD +short ${REVNET}.dnsbl.dronebl.org` || (echo 0; return 1)
	if [[ "$DIG_OUT" == "" ]]; then (echo 0; return 0); fi
	for i in $DIG_OUT; do 
		DNSBL_CODE=`echo $i | (IFS='.' read a b c d; echo "$d")`
		case $DNSBL_CODE in 
			6 | 7 | 13 | 15 | 16)
				echo 8
				return 0
			;;
		esac
	done
	echo 0
	return 1
}

update_expiry()
{
	local TIMEunit
	local MULTI

	if [[ "$1" == ""  ]]; then
		return 1
	fi
	MULTI=`printf %d $1`
	if [[ "$MULTI" == "" ]]; then
		return 1
	fi

	local -a AEXPIRE=(${DEFAULT_EXPIRY})

	DBLMULTI="`drone_bl_expiry`"
	if [[ "$DBLMULTI" != "" ]]; then
		DBLMULTI="`printf %d $DBLMULTI`"
		if [[ "$DBLMULTI" != "" ]]; then
			((MULTI += DBLMULTI))
		fi
	fi

	if [[ "${MAX_EXPIRY}" == "" ]] && [[ "${MAXEXPIRE}" != "" ]]; then
		if (( MAXEXPIRE <= MULTI )); then
			MULTI=$MAXEXPIRE
		fi
	fi


	SELECT_TXT="select \"true\" where ((datetime(\"${AOUTPUT[expires]}\") > datetime(CURRENT_TIMESTAMP,- \"1 DAYS\")) "
	SELECT_TXT="$SELECT_TXT or (\"${AOUTPUT[stamp]}\" < datetime(\"now\",- \"${RECIDIVE_FORCEACT}\" )));"
	local SQLOUT
	local UPDATE_TXT
	SQLOUT=`echo $SELECT_TXT | $SQLITE_CMD` || return $?
	if [[ "$SQLOUT" != "" ]]; then
		((MULTI *= 2 ))	
	fi

	if [[ "${AEXPIRE[0]}" != "" ]]; then
		TIMEunit=`printf %d "${AEXPIRE[0]}"`	
		if [[ "${TIMEunit}" != "" ]]; then
		
			((TIMEunit *= MULTI))
			TIMEunit=`printf %d $TIMEunit`
			if [[ "$TIMEunit" != "" ]]; then
				DEFAULT_EXPIRY="$TIMEunit ${AEXPIRE[1]}"
			fi
		fi
	else
		return 1
	fi

	if [[ "${MAX_EXPIRY}" != "" ]]; then
		SELECT_TXT="select \"true\" where ((datetime(CURRENT_TIMESTAMP,+ \"${MAX_EXPIRY}\")"
		SELECT_TXT="$SELECT_TXT < (datetime(CURRENT_TIMESTAMP,+ \"${DEFAULT_EXPIRY}\"))));"
		SQLOUT=`echo $SELECT_TXT | $SQLITE_CMD` || return $?
		if [[ "$SQLOUT" != "" ]]; then
			DEFAULT_EXPIRY="${MAX_EXPIRY}"
		fi
	fi

	return 0
}


rtbh_db_refresh()
{
	WHERE_TXT="active = 1 and added = 1 and datetime(expires) > datetime(CURRENT_TIMESTAMP)"
	ORDER_TXT="order by expires DESC"
	LIMIT_TXT="LIMIT 1"
	rtbh_db_select
	
	if [[ "${AOUTPUT[seq]}" != "" ]]; then
		REFRESH_EXPIRY="expires = datetime(strftime(\"%s\",\"${AOUTPUT[expires]}\") + \
		(strftime(\"%s\",'now' )-strftime(\"%s\",\"${AOUTPUT[stamp]}\")),'unixepoch')"
		INSERT_TEXT="UPDATE rtbh set $SQL_STAMP, active = 1, added = 1, deleted = 0,"
		INSERT_TEXT="$INSERT_TEXT $REFRESH_EXPIRY" 
		INSERT_TEXT="$INSERT_TEXT where seq=${AOUTPUT[seq]};"
		echo "$INSERT_TEXT" | $SQLITE_CMD
		return $?
	else
		rtbh_db_add
	fi
	return $?
}

rtbh_geoip_lookup_cc()
{
	local netcc
	if [[ "$1" == "" ]]; then return 1; fi
	netcc=`$GEOIPLOOKUP $1 | grep "ountry" | cut -f2 -d: | cut -f1 -d, | tr -d '[[:space:]]'` 
	if [[ "${#netcc}" -eq 2 ]]; then
		echo $netcc
		return 0
	fi
	return 1
}

rtbh_db_add_geoip_data()
{
	rtbh_geoip_enabled || return 1
	unset RTBH_GEOIP_COLS
	unset RTBH_GEOIP_DATA
	NETCC=`rtbh_geoip_lookup_cc $NETWORK`
	if [[ "$NETCC" != "" ]]; then
		RTBH_GEOIP_COLS=" , geoip_cc "
		RTBH_GEOIP_DATA=" , \"${NETCC}\" "
		return 0
	fi
	return 1
}

rtbh_db_add_geoip_expiry()
{
(
	if [[ "$NETCC" == "" ]] && [[ "$1" != "" ]] && rtbh_geoip_enabled; then
		NETCC=`rtbh_geoip_lookup_cc $1`
	fi
	NETCC=${NETCC:-"=="}

	CCLINE=`grep -v "^#" < "$GEOIPCONF" | grep "^${NETCC};.*;.*;32;.*;[Xx]$\|^__;.*;.*;32;.*;[Xx]$" || return 1`
	if [[ "$CCLINE" == "" ]]; then return 1; fi
	CCEXPIRY=`echo $CCLINE | cut -f6 -d\;`
	CCMEXPIRY=`echo $CCLINE | cut -f7 -d\;`
	SELECT_TXT="select min(strftime(\"%s\", \"now\", \"+$DEFAULT_EXPIRY\", "
	SELECT_TXT="$SELECT_TXT \"+`parse_time_arg $CCEXPIRY`\"), "
	SELECT_TXT="$SELECT_TXT (strftime(\"%s\", \"now\", \"+`parse_time_arg $CCMEXPIRY`\")))"
	SELECT_TXT="$SELECT_TXT - (strftime(\"%s\",\"now\")) as EXPIRE"
	OUTPUT=`echo "$SELECT_TXT ; " | $SQLITE_CMD || return 1`
	OUTPUT=`echo $OUTPUT | cut -f2 -d= -s`
	if [[ "$OUTPUT" == "" ]]; then return 1; fi
	OUTPUT=`printf %d $OUTPUT || return 1`
	if [[ "$OUTPUT" != "" ]] && [[ "$OUTPUT" != "0" ]]; then 
		echo "$OUTPUT seconds"
	else
		return 0
	fi
	return 0
)
}

rtbh_db_add()
{
	ORDER_TXT="order by stamp DESC"
	LIMIT_TXT="LIMIT 1"
	rtbh_db_select 
	
	if [[ "${AOUTPUT[seq]}" != "" ]]; then
		update_expiry `rtbh_recidive_wide` || update_expiry ${AOUTPUT[times]}
		INSERT_TEXT="UPDATE rtbh set $SQL_STAMP, active = 1, added = 1, deleted = 0,"
		INSERT_TEXT="$INSERT_TEXT times = ${AOUTPUT[times]}+1, expires = `rtbh_expire_time`" 
		if [[ "${AOUTPUT[active]}" != "1" ]]; then
			INSERT_TEXT="$INSERT_TEXT , fullstring = \"${FS[@]}\""
		fi
		INSERT_TEXT="$INSERT_TEXT where seq=${AOUTPUT[seq]};"
		echo "$INSERT_TEXT" | $SQLITE_CMD
		return $?
	fi
	rtbh_db_add_geoip_data && [[ "$SUBNET" == "/32" ]] &&\
	GEOIP_EXPIRY="`rtbh_db_add_geoip_expiry`" && DEFAULT_EXPIRY="$GEOIP_EXPIRY"
	INSERT_TEXT="INSERT into 'rtbh' (route,mask,lowner,rowner,fullstring,times,active,expires,added $RTBH_GEOIP_COLS) values"
	INSERT_TEXT="$INSERT_TEXT (\"$NETWORK\",\"$SUBNET\",\"$USER$USERFROM\",\"$ROWNER\",\"${FS[@]}\",1,1,"
	INSERT_TEXT="$INSERT_TEXT (`rtbh_expire_time`),1 $RTBH_GEOIP_DATA);"
	echo "$INSERT_TEXT" | $SQLITE_CMD
	return $?
	
}

rtbh_recidive_del()
{
	unset RECWIDETIME
	local RECWIDETIME

	if [[ "${#AOUTPUT[@]}" -lt 4 ]] || [[ "${AOUTPUT[times]}" == "" ]]; then
		return 0
	fi

	if [[ "${RECIDIVE_TIMES}" == "" ]]; then return 0; fi
	
	if ((AOUTPUT[times] < RECIDIVE_TIMES)); then 
		RECWIDETIME=`rtbh_recidive_wide` || return 0
	fi

	SELECT_TXT="select \"true\" where ((datetime(\"${AOUTPUT[expires]}\") < datetime(CURRENT_TIMESTAMP, + \"${RECIDIVE_FORCEACT}\")) "
	SELECT_TXT="$SELECT_TXT or (\"${AOUTPUT[stamp]}\" < datetime(\"now\",- \"${RECIDIVE_FORCEACT}\" )));"
	
	local SQLOUT
	local UPDATE_TXT
	SQLOUT=`echo $SELECT_TXT | $SQLITE_CMD` || return $?
	if [[ "$SQLOUT" != "" ]]; then
		return 0
	fi
	update_expiry ${RECWIDETIME:-${AOUTPUT[times]}}
	LONGEST_ACTIVE=${RECIDIVE_ACTIVE}

	UPDATE_TXT="UPDATE rtbh set expires = min(datetime(\"${AOUTPUT[expires]}\"),`rtbh_expire_time`) where seq = ${AOUTPUT[seq]};"
	echo "$UPDATE_TXT" | $SQLITE_CMD || return $?
	return 1
}

rtbh_rint()
{

	if [[ "$UID" != "0" ]]; then return 1; fi
	if [[ "$RTBH_CRON_JOB" == "1" ]]; then return 1; fi
	if tty -s; then return 0; fi
	if [[ "$PS1" != "" ]]; then return 0; fi
	if -t 0; then return 0; fi
	if [[ "$-" == "*i*" ]]; then return 0; fi
	return 1

}

rtbh_db_del()
{
	rtbh_db_check 
	if [[ "$?" == "1" ]]; then return 0; fi
	if [[ "$?" == "2" ]]; then return 3; fi
	if [[ "$?" -gt "2" ]]; then return $?; fi

	while ! rtbh_rint; do

	if [[ "${#AOUTPUT[@]}" -lt 4 ]] || [[ "${AOUTPUT[active]}" == "0" ]]; then
		return 0
	fi
	
	if [[ "$ROWNER" == "" ]]; then
		if [[ "${USER}${USERFROM}" != "${AOUTPUT[lowner]}" ]]; then
			return 2
		fi
	elif [[ "$ROWNER" != "${AOUTPUT[rowner]}" ]]; then
		return 2
	fi
	if [[ "$FS_WHERE_NAME" != "" ]] ; then
		echo "${AOUTPUT[fullstring]}" | grep -E -q "[[:space:]]${FS_WHERE_NAME}[[:space:]]" || return 2
	fi
	if [[ "${db_mode}" != "forcedel" ]]; then
		rtbh_recidive_del
		if [[ "$?" == "1" ]]; then return 2; fi
		if [[ "$?" == "2" ]]; then return 3; fi
		if [[ "$?" -gt "2" ]]; then return $?; fi
	fi
	break
	done
	
	echo "UPDATE rtbh set active = 0, deleted = 1, $SQL_STAMP WHERE seq = ${AOUTPUT[seq]};" | $SQLITE_CMD
	return 0
}

rtbh_db_update_exp()
{
	SELECT_TXT="select \"true\" from rtbh_status where (datetime(stamp) > datetime(CURRENT_TIMESTAMP, \"-1 MINUTES\"));"
	SQLOUT="`echo $SELECT_TXT | $SQLITE_CMD`" || return $?
	if [[ "$SQLOUT" != "" ]]; then
		return 0
	fi
	UPDATE_TXT="UPDATE rtbh set active = 0, $SQL_STAMP  WHERE `rtbh_debug_where` active = 1 and (datetime(expires) < datetime(CURRENT_TIMESTAMP)"
	UPDATE_TXT="$UPDATE_TXT or datetime(stamp) < datetime(CURRENT_TIMESTAMP, \"-$LONGEST_ACTIVE\"));"
	echo "$UPDATE_TXT" | $SQLITE_CMD || return $?
	DELETE_TXT="DELETE from rtbh WHERE active = 0 and datetime(expires) < datetime(CURRENT_TIMESTAMP, \"-$LONGEST_KEEP\");"
	echo "$DELETE_TXT" | $SQLITE_CMD || return $?
	DELETE_TXT="DELETE from rtbh WHERE active = 0 and datetime(stamp) < datetime(CURRENT_TIMESTAMP, \"-$LONGEST_KEEP\");"
	echo "$DELETE_TXT" | $SQLITE_CMD || return $?
	UPDATE_TXT="update rtbh_status set stamp = CURRENT_TIMESTAMP where stamp < CURRENT_TIMESTAMP;"
	echo "$UPDATE_TXT" | $SQLITE_CMD || return $?
	return $?

}

rtbh_db_delete_exp()
{
	if [[ ! -x "`which $1`" ]]; then return 1; fi
	i=0
	SELECT_TXT="select route,mask,fullstring,seq from rtbh where `rtbh_debug_where` active = 0 and deleted = 0 "
	SELECT_TXT="$SELECT_TXT and stamp > datetime(CURRENT_TIMESTAMP, \"-$LONGEST_ACTIVE\")"
	SELECT_TXT="$SELECT_TXT GROUP BY route,mask order by stamp DESC limit 1" 
	CHK_SELECT_TXT="select route,mask from rtbh"
	CHK_SELECT_TXT="$CHK_SELECT_TXT where active = 1 and deleted = 0 and datetime(expires) > CURRENT_TIMESTAMP"
	while OUTPUT=`echo "$SELECT_TXT offset $i;" | $SQLITE_CMD`; do
		if [[ "$OUTPUT" == "" ]]; then break; fi
		sqlout_to_aar AOUTPUT OUTPUT || return $?
		WHERE_TXT="where route = \"${AOUTPUT[route]}\" and mask = \"${AOUTPUT[mask]}\" and datetime(expires) < CURRENT_TIMESTAMP;"
		echo "update rtbh set deleted = 1 $WHERE_TXT" | $SQLITE_CMD || return $?
		CHKL_SELECT_TXT="$CHK_SELECT_TXT and route = \"${AOUTPUT[route]}\" and mask = \"${AOUTPUT[mask]}\""
		OUTPUT=`echo "$CHKL_SELECT_TXT order by stamp DESC limit 1;" | $SQLITE_CMD || continue`
		if [[ "$OUTPUT" != "" ]]; then 
			((i++));
			continue; 
		fi
		(RTBH_DB="$0" $1 ${AOUTPUT[route]}${AOUTPUT[mask]} ${AOUTPUT[fullstring]} 2>&1 >/dev/null)
	done
	return 0
}


rtbh_db_clean()
{
	if [[ ! -x "`which $1`" ]]; then return 1; fi
	i=0
	SELECT_TXT="select route,mask,seq from rtbh where active = 1 and deleted = 0 "
	SELECT_TXT="$SELECT_TXT and datetime(stamp) < datetime(CURRENT_TIMESTAMP, \"-$LONGEST_ACTIVE\")"
	SELECT_TXT="$SELECT_TXT and datetime(expires) < datetime(CURRENT_TIMESTAMP)"
	SELECT_TXT="$SELECT_TXT GROUP BY route,mask order by stamp DESC limit 1" 
	UPDATE_TXT="update rtbh set active = 0, deleted = 1"
	while OUTPUT=`echo "$SELECT_TXT offset $i;" | $SQLITE_CMD`; do
		if [[ "$OUTPUT" == "" ]]; then break; fi
		sqlout_to_aar AOUTPUT OUTPUT || return $?
		if (RTBH_DB="$0" $1 ${AOUTPUT[route]}${AOUTPUT[mask]} 2>&1 /dev/null); then
			WHERE_TXT="where seq = \"${AOUTPUT[seq]}\";"
			echo "$UPDATE_TXT $WHERE_TXT" | $SQLITE_CMD || return $?
			continue	
		fi
		WHERE_TXT="where route = \"${AOUTPUT[route]}\" and mask = \"${AOUTPUT[mask]}\";"
		echo "$UPDATE_TXT $WHERE_TXT" | $SQLITE_CMD || return $?
	done
	return 0
}

rtbh_db_route_sync()
{

	if [[ ! -x "`which $1`" ]]; then return 1; fi
	if [[ ! -x "`which $2`" ]]; then return 1; fi
	
	SELECT_TXT="select route,mask from rtbh"
	WHERE_TXT="where `rtbh_debug_where` active = 1"
	WHERE_TXT="$WHERE_TXT and deleted = 0 and datetime(expires) > datetime(CURRENT_TIMESTAMP)" 
	CHK_WHERE="$WHERE_TXT and deleted = 1 or datetime(expires) < datetime(CURRENT_TIMESTAMP)"
	ORDER_TXT="order by 'stamp' desc limit 1"

	(RTBH_DB="$0" $1) | while read IN; do 
		if [[ "$IN" == "" ]] || [[ "${IN:1}" == "#" ]]; then continue; fi
		rtbh_set_net_mask $IN || continue
		WHERE_NET="and route = \"$NETWORK\" and mask = \"/$SUBNETBITS\""
		OUTPUT=`echo "$SELECT_TXT $WHERE_TXT $WHERE_NET $ORDER_TXT;" | $SQLITE_CMD`
		if [[ "$OUTPUT" == "" ]]; then 
			OUTPUT=`echo "$SELECT_TXT $CHK_WHERE $WHERE_NET $ORDER_TXT;" | $SQLITE_CMD`
			if [[ "$OUTPUT" != "" ]]; then
				if [[ -x "`which $3`" ]]; then
					(RTBH_DB="$0" $3 ${NETWORK}${SUBNET})
				fi
				continue
			fi
			(RTBH_DB="$0" $2 ${NETWORK}${SUBNET})
		fi
	done
}

rtbh_db_sync()
{
	if [[ ! -x "`which $1`" ]]; then return 1; fi

	SELECT_TXT="select route,mask,fullstring,seq from rtbh where `rtbh_debug_where` active "
	CHK_SELECT_TXT="$SELECT_TXT != $2 and datetime(expires) > datetime(CURRENT_TIMESTAMP)" 
	SELECT_TXT="$SELECT_TXT = $2 and "


	EXPIRE_TXT=""

	if [[ "$2" == "0" ]]; then
		UPDATE_TXT="UPDATE rtbh set deleted = 1, added = 0 "
		if [[ "$FULL_SYNC" != "yes" ]]; then
			EXPIRE_TXT="deleted = 0 and "
		fi
		EXPIRE_TXT="${EXPIRE_TXT} datetime(expires) > datetime(CURRENT_TIMESTAMP,\"-$LONGEST_ACTIVE\")" 
	elif [[ "$2" == "1" ]]; then
		UPDATE_TXT="UPDATE rtbh set added = 1, deleted = 0 "
		if [[ "$FULL_SYNC" != "yes" ]]; then
			EXPIRE_TXT="added = 0 and "
		fi
		EXPIRE_TXT="${EXPIRE_TXT} datetime(expires) > datetime(CURRENT_TIMESTAMP)" 
	else
		return 1
	fi
	GROUP_TXT="GROUP by route,mask"
	ORDER_TXT="order by stamp ASC limit 1"
	SELECT_TXT="$SELECT_TXT $EXPIRE_TXT $GROUP_TXT $ORDER_TXT"

	i=0
	while OUTPUT=`echo "$SELECT_TXT offset $i;" | $SQLITE_CMD`; do
		if [[ "$OUTPUT" == "" ]]; then break; fi
		sqlout_to_aar AOUTPUT OUTPUT || return $?
		WHERE_TXT="route = \"${AOUTPUT[route]}\" and mask = \"${AOUTPUT[mask]}\""
		CHKL_SELECT_TXT="$CHK_SELECT_TXT and $WHERE_TXT order by stamp DESC limit 1"
		OUTPUT=`echo "$CHKL_SELECT_TXT ;" |  $SQLITE_CMD`
		if [[ "$OUTPUT" != "" ]]; then 
			((i++))
			continue
		fi	
		(RTBH_DB="$0" $1 ${AOUTPUT[route]}${AOUTPUT[mask]} ${AOUTPUT[fullstring]})
		echo "$UPDATE_TXT where $WHERE_TXT ;" | $SQLITE_CMD
		if [[ "$?" != "0" ]] || [[ "$FULL_SYNC" == "yes" ]]; then
			((i++))
		fi
	done
	return 0
}

rtbh_dump()
{
	shift
	local SELECT_COUNT=""
	local SELECT_TXT='select ${DISTINCT_TXT}route,mask from rtbh'
	local ORDER_TXT="order by stamp DESC limit 1"
	local WHERE_TXT="where stamp > datetime(CURRENT_TIMESTAMP,\"-$LONGEST_KEEP\")"
	local	i=0
	local	j=0
	while [[ "$1" != "" ]]; do
	case "$1" in 
	
		all)
			;;	
		not)
			if [[ "$2" != "" ]]; then
				NOTB="(not"
				NOTE=")"
				shift
				continue
			else
				echo "expecting more arguments"
			fi
			;;
		mask | subnet)
			if [[ "$2" != "" ]]; then
				MASKA=$2
				WHERE_TXT="$WHERE_TXT and $NOTB mask=\"$2\" $NOTE"
			else
				echo "expecting a mask in slash notation"
				return 1
			fi
			shift
			;;
			
		prefix | route)
			MATCH="="
			MATCHE=""
			if [[ "${2,,}" == "partial" ]]; then
				MATCH="like"
				MATCHE="%"
				shift
			fi
			if [[ "$2" != "" ]]; then
				MASK="`echo $2 | cut -s -f2 -d'/'`"
				PREFIX="`echo $2 | cut -f1 -d'/'`"
				if [[ "${PREFIX}" == "" ]]; then PREFIX=$2 ; fi
				if [[ "$MASK" != "" ]]; then
					WHERE_TXT="$WHERE_TXT and $NOTB mask=\"/$MASK\" $NOTE"
				fi
				SELECT_TXT="select ${DISTINCT_TXT}route,mask from rtbh"
				WHERE_TXT="$WHERE_TXT and $NOTB route $MATCH \"${PREFIX}${MATCHE}\" $NOTE"
			else
				echo "expecting a prefix/ip number"
				return 1
			fi
			shift
			;;
		count)
			SELECT_COUNT="count"
			;;
		times)
			if [[ "$2" != "" ]]; then
				WHERE_TXT="$WHERE_TXT and $NOTB times >= $2 $NOTE"
			else
				echo "expecting a number"
				return 1
			fi	
			shift
			;;
		checkactive)
			if [[ "$2" == "" ]] || [[ ! -x "$2" ]]; then
				echo "This dump mode expects a check command"
				return 1
			fi
			if [[ "$NOTB" != "" ]]; then
				eval "$(cat <<'EOF'
_not_checkcmd()
{ 
$* && return 1
 return 0
}
EOF
)"
				CHKCMD="_not_checkcmd $2"
			else
				CHKCMD="$2"
			fi
			shift
			unset NOTB
			unset NOTE
			;&
		active)
			WHERE_TXT="$WHERE_TXT and $NOTB active = 1 $NOTE"
			;;
		inactive)
			WHERE_TXT="$WHERE_TXT and $NOTB active = 0 $NOTE"
			;;
		older)
			NEOLD="<"
			;&
		newer)	
			NEOLD=${NEOLD:-">"}
			if [[ "$2" != "" ]]; then
				WHERE_TXT="$WHERE_TXT and $NOTB datetime(stamp) $NEOLD datetime(CURRENT_TIMESTAMP, \"-`parse_time_arg \"$2\"`\") $NOTE"
				shift
				unset NEOLD
			else
				echo "This dump mode expects a time value as argument (5:days)"
				return 1
			fi
			;;
		expires)
			if [[ "$2" != "" ]]; then
				WHERE_TXT="$WHERE_TXT and $NOTB datetime(expires) < datetime(CURRENT_TIMESTAMP, \"+`parse_time_arg \"$2\"`\") $NOTE"
				shift
				unset NEOLD
			else
				echo "This dump mode expects a time value as argument (5:days)"
				return 1
			fi
			;;

		added)
			WHERE_TXT="$WHERE_TXT and $NOTB added = 1 $NOTE"
			;;
		deleted)
			WHERE_TXT="$WHERE_TXT and $NOTB deleted = 1 $NOTE"
			;;
		expired)
			WHERE_TXT="$WHERE_TXT and $NOTB datetime(expires) < datetime(CURRENT_TIMESTAMP) $NOTE"
			;;
		ancient)
			WHERE_TXT="where $NOTB datetime(stamp) < datetime(CURRENT_TIMESTAMP,\"-$LONGEST_ACTIVE\") $NOTE"
			;;
		limit)
			if [[ "$2" != "" ]]; then
				LIMIT="`printf %d $2`"
				if [[ "$LIMIT" == "" ]]; then
					echo "This dump mode expects a numeric argument"
				fi
			fi
			shift
			;;
		text)
			if [[ "$2" == "" ]]; then
				echo "This dump mode expects a string argument"
				return 1
			fi
			WHERE_TXT="$WHERE_TXT and $NOTB fullstring like \"%$2%\" $NOTE"
			shift
			;;
		lowner | rowner)
			if [[ "$2" == "" ]]; then
				echo "This dump mode expects a string argument"
				return 1
			fi
			WHERE_TXT="$WHERE_TXT and $NOTB $1 like \"%$2%\" $NOTE"
			shift
			;;
		geoip)

			if [[ "$2" == "" ]]; then
				echo "This dump mode expects either a CC code or the top # argument"
				return 1
			fi
			if [[ "${2,,}" == "top" ]]; then
				shift
				WHERE_TXT="${WHERE_TXT} and geoip_cc != \"\""
				rtbh_db_geoip_top $2
				return 0
			fi
			WHERE_TXT="$WHERE_TXT and $NOTB geoip_cc like \"%$2%\" $NOTE"
			shift
			;;

		cidr)
			if ! sqlite_inet_enabled; then
				 echo "this dump mode is not available"
				 return 1
			fi
			CIDRA=(`v4dec $2`)
			if [[ "${CIDRA[0]}" == "" ]]; then
				echo "This dump mode expects a subnet with masklen"
				return 1
			fi
			CIDRA[0]=$((CIDRA[0] - (CIDRA[0] % ${CIDRA[1]:1})))
			WHERE_TXT="$WHERE_TXT and $NOTB ((inet_aton(route) BETWEEN ${CIDRA[0]} and "
			WHERE_TXT="$WHERE_TXT ${CIDRA[0]} + ${CIDRA[1]:1} )"
			WHERE_TXT="$WHERE_TXT and ( (1<<32 - cast(substr(mask,2,2) as INTEGER)) <"
			WHERE_TXT="$WHERE_TXT (${CIDRA[1]:1}+1) )"
			WHERE_TXT="$WHERE_TXT ) $NOTE"

			shift
			;;
			
		format)
			case "$2" in 
				full)
					;;
				string)
					OUTPUTSTRING=1
					;;
				net)
					OUTPUTNET=1
					;;
				raw)
					OUTPUTRAW=1
					;;
				*)
					echo "format supports types full, net, raw, string"
					return 1
					;;
			esac
			shift
			;;
		distinct)
			DISTINCT_TXT="distinct seq,"
			;;
			
		*)
			echo "This command mode expect either of all, active, inactive, added, deleted, expired, expires [timeval], ancient, text string, limit number, count, not, format type, older, newer, prefix|route [partial] , cidr, distinct , mask, geoip [ CC, top (number) ], lowner, rowner"
			return 1
		;;
		
	esac
	shift
	unset NOTB
	unset NOTE
	done
	eval SELECT_TXT=\"${SELECT_TXT}\"
	if [[ "$SELECT_COUNT" != "" ]]; then
		OUTPUT=`echo "select count(all) from ($SELECT_TXT $WHERE_TXT);" | $SQLITE_CMD`
		if [[ "$OUTPUT" == "" ]]; then return $?; fi
		printf "%d\n" `echo $OUTPUT | cut -f2 -d'='`
		return 0
	fi
	if [[ "$OUTPUTRAW" == "1" ]]; then
		unset LIMIT_TXT
		if [[ "$LIMIT" != "" ]]; then
			LIMIT_TXT="limit $LIMIT"
		fi
		if [[ "$DISTINCT_TXT" != "" ]]; then
			echo "select distinct route,mask from ($SELECT_TXT $WHERE_TXT ORDER by 'stamp' desc) $LIMIT_TXT ;" | (DO_COPROC=0 SQLITE_ARGS="-list" $SQLITE_CMD) | tr -d '\|'
			return 0
		fi
		echo "$SELECT_TXT $WHERE_TXT ORDER by 'stamp' desc $LIMIT_TXT ;" | (DO_COPROC=0 SQLITE_ARGS="-list" $SQLITE_CMD) | tr -d '\|'
		return 0
	fi
		
	while OUTPUT=`echo "$SELECT_TXT $WHERE_TXT $ORDER_TXT offset $i;" | $SQLITE_CMD`; do
		if [[ "$OUTPUT" == "" ]]; then break; fi
		if [[ "$LIMIT" != "" ]] && ((i==LIMIT)); then
			break
		fi
		sqlout_to_aar AOUTPUT OUTPUT || return $?
		((i++))
		if [[ "$CHKCMD" != "" ]]; then
			(RTBH_DB="$0" $CHKCMD ${AOUTPUT[route]}${AOUTPUT[mask]} >/dev/null)
			if [[ "$?" == 0 ]]; then
				echo "${AOUTPUT[route]}${AOUTPUT[mask]} active"
			else
				echo "${AOUTPUT[route]}${AOUTPUT[mask]} inactive"
			fi
			continue	
		fi
		qstring="select \* from rtbh $WHERE_TXT and route = \"${AOUTPUT[route]}\""
		qstring="$qstring and mask = \"${AOUTPUT[mask]}\""
		if [[ "${AOUTPUT[seq]}" != "" ]]; then
			qstring="$qstring and seq = \"${AOUTPUT[seq]}\""
		fi
		OUTPUT=`echo "$qstring $ORDER_TXT ;" | $SQLITE_CMD`
		if [[ "$OUTPUT" == "" ]]; then continue; fi
		sqlout_to_aar AOUTPUT OUTPUT || continue
		local DB_out=""
		if [[ "$OUTPUTNET" == "1" ]]; then
			DB_out="${AOUTPUT[route]}${AOUTPUT[mask]}"
		fi
		if [[ "$OUTPUTSTRING" == "1" ]]; then 
			if [[ "$DB_out" != "" ]]; then DB_out="$DB_out "; fi
			DB_out="${DB_out} ${AOUTPUT[fullstring]}"
		fi
		if [[ "$DB_out" != "" ]]; then 
			echo $DB_out
			continue
		fi

		for j in "${!AOUTPUT[@]}"; do
			if [[ "$DB_out" != "" ]]; then 
				DB_out="${DB_out}${DUMP_DELIM}" 
			fi
			DB_out="${DB_out}$j=${AOUTPUT[$j]}"
		done
		echo "$DB_out"
	done
	return 0


}

rtbh_db_geoip_top()
{
	rtbh_geoip_enabled || return 1
	if [[ "$1" != "" ]]; then 
		LIMIT=`printf %d $1`
		if [[ "$LIMIT" != "" ]]; then
			LIMIT_TXT="limit $LIMIT"
		fi
	fi
	i=0
	SELECT_TXT="select geoip_cc, count(geoip_cc) as num from rtbh"
	WHERE_TXT=${WHERE_TXT:-"where active = 1 and  stamp > datetime(CURRENT_TIMESTAMP,\"-$LONGEST_KEEP\")"}
	GROUP_TXT="group by geoip_cc order by num DESC "
	while OUTPUT=`echo "$SELECT_TXT $WHERE_TXT $GROUP_TXT LIMIT 1 offset $i;" | $SQLITE_CMD`; do
		if [[ "$OUTPUT" == "" ]]; then break; fi
		if [[ "$LIMIT" != "" ]] && ((i==LIMIT)); then
			break
		fi
		sqlout_to_aar AOUTPUT OUTPUT || return $?
		((i++))
		NETCC=${AOUTPUT[geoip_cc]:-"=="}
		echo -e "${AOUTPUT[geoip_cc]}\t${AOUTPUT[num]}"
	done
		
}

rtbh_db_geoip_report()
{
	GEOIP_REPORT="1"
	RTBH_ROUTE="echo"
	rtbh_db_geoip_add $*
	return 0
}

rtbh_db_geoip_check()
{

	if [[ "$1" != "" ]]; then rtbh_set_net_mask $1; fi
	FND=`rtbh_dump dump route $NETWORK mask $SUBNET active text $GEOIPSTRING format raw limit 1 ${TIME}` || return 1
	if [[ "$FND" != "" ]]; then return 0; fi
	return 1
}

rtbh_db_geoip_add()
{
	shift
	rtbh_set_net_mask $*
	if [[ "$NETWORK" == "" ]]; then return 1; fi
	NETCC=`rtbh_geoip_lookup_cc $NETWORK`
	#notfound-default?
	NETCC=${NETCC:-"=="}
	OLDIFS="$IFS"
	IFS="${IFS}."
	NETARR=($NETWORK)
	IFS="$OLDIFS"
	unset GEOIPTEXTLINE
	GEOIPNOTTEXTLINE=`for i in $GEOIPNOTNAMES; do echo -ne " not text $i"; done`
	for i in $GEOIPNAMES; do
		if [[ "$GEOIPTEXTLINE" != "" ]]; then GEOIPTEXTLINE="$GEOIPTEXTLINE or"; fi
		GEOIPTEXTLINE="$GEOIPTEXTLINE text $i"
	done
	
	CCLINE_TYPES="bB${GEOIP_CHILD:+cC}"
	grep -v "^#" < "$GEOIPCONF" | grep "^${NETCC};.*;[${CCLINE_TYPES}]$\|^__;.*;[${CCLINE_TYPES}]$" |\
	while read CCLINE; do
		CCNUM=`echo $CCLINE | cut -f2 -d\;`
		CCTIME=`echo $CCLINE | cut -f3 -d\;`
		CCBITS=`echo $CCLINE | cut -f4 -d\;`
		CCBITS=`printf %d $CCBITS 2>/dev/null`
		CCCLASS=`echo $CCLINE | cut -f5 -d\;`
		CCEXPIRY=`echo $CCLINE | cut -f6 -d\;`
		CCMEXPIRY=`echo $CCLINE | cut -f7 -d\;`
		if [[ "$CCMEXPIRY" != "" ]]; then
			CCMEXTXT="maxexpire $CCMEXPIRY"
		fi
		if [[ "$CCTIME" != "" ]] ; then 
			CCTIME="newer $CCTIME"
		else
			CCTIME="newer 30:min"
		fi
		CCEXPIRY="expires ${CCEXPIRY:-$GEOIPDEFEXPIRY}"
		
		unset PREFIX
		unset ROUTE
		case "${CCCLASS,,}" in
			a)
				PREFIX="${NETARR[0]}/32"
				ROUTE="${NETARR[0]}.0.0.0/8"
				CLASSBITS=8
				;;
			b)
				PREFIX="${NETARR[0]}.${NETARR[1]}/32"
				ROUTE="${NETARR[0]}.${NETARR[1]}.0.0/16"
				CLASSBITS=16
				;;
			c)
				PREFIX="${NETARR[0]}.${NETARR[1]}.${NETARR[2]}/32"
				ROUTE="${NETARR[0]}.${NETARR[1]}.${NETARR[2]}.0/24"
				CLASSBITS=24
				;;
			*)
				echo "invalid class: $CCLINE"
				continue
				;;
		esac
				
#		rtbh_db_dump  distinct not text dns format raw newer 1:hour mask /32 prefix partial 180.76/32 count 
		FND=0
		evalFS='FS=("${GEOIP_CHILD:+child ${GEOIP_CHILD} }"$GEOIPSTRING ""${FND:+found ${FND} }"from $NETWORK$SUBNET for $CCLINE" $CCMEXTXT $CCEXPIRY)'
		if [[ "$CCBITS" != "" ]] && sqlite_inet_enabled; then
			if [[ "$SUBNETBITS" != "32" ]] && [[ "$CCBITS" -lt "$SUBNETBITS" ]]; then 
				if [[ "$SUBNETBITS" -gt "$CLASSBITS" ]]; then
					continue;
				else
					CCBITS="$CLASSBITS"
				fi
			fi
			CIDRA=(`v4dec ${NETWORK}/${CCBITS}`)
			CIDRA[0]=$((CIDRA[0] - (CIDRA[0] % ${CIDRA[1]:1})))
			ROUTE="`inet_ntoa ${CIDRA[0]}`/${CCBITS}"
			( TIME=$CCTIME rtbh_db_geoip_check $ROUTE && eval $evalFS && rtbh_db_refresh ) && continue
			FND=`rtbh_dump dump distinct $GEOIPNOTTEXTLINE format raw $CCTIME cidr ${ROUTE} count`
		else
			( TIME=$CCTIME rtbh_db_geoip_check $ROUTE && eval $evalFS && rtbh_db_refresh ) && continue 
			FND=`rtbh_dump dump distinct $GEOIPNOTTEXTLINE format raw $CCTIME mask /32 prefix partial $PREFIX count`
		fi
		FND=`printf %d $FND`
		if [[ "$FND" != "" ]] && [[ "$FND" -ge "$CCNUM" ]]; then
			eval $evalFS
			$RTBH_ROUTE $ROUTE ${FS[@]} 
			if [[ "$?" == "0" ]]; then return 0; fi
		fi
	done
	return 1
}

rtbh_geoip_catchup()
{
	catchupSORT="sort | uniq"

	while [[ "$1" != "" ]]; do
	case "${1,,}" in
		time)
			shift
			TIME="1:hour"
			if [[ "$1" != "" ]]; then
				TIME="$1"
				shift
			fi
			TIME_TXT="newer $TIME"
			;;
		limit)
			shift
			if [[ "$1" != "" ]]; then
				LIMIT=`printf %d $1`
				shift
			fi
			LIMIT="${LIMIT:-250}"
			LIMIT_TXT="limit $LIMIT"
			;;
		total)
			shift
			if [[ "$1" != "" ]]; then
				TOTAL=`printf %d $1`
				shift
			fi
			TOTAL="${TOTAL:-25}"
			;;
		bits)
			shift
			if [[ "$1" != "" ]]; then
				catchupBITS=`printf %d $1`
				shift
			fi
			catchupBITS="${catchupBITS:-24}"
			catchupSORT="v4decpipe | sort -n -k1 | uniq "
			;;
		*)
			if [[ "$1" != "" ]]; then 
				echo "This geoip catchup mode expects as argument either time [time-def], limit [record-limit], total [total-blocks], not $1"
				exit 1
			fi
	esac
	done

	GEOIPNOTTEXTLINE=`for i in $GEOIPNOTNAMES; do echo -ne " not text $i"; done`
	for i in $GEOIPNAMES; do
		if [[ "$GEOIPTEXTLINE" != "" ]]; then GEOIPTEXTLINE="$GEOIPTEXTLINE or"; fi
		GEOIPTEXTLINE="$GEOIPTEXTLINE text $i"
	done

	i=0
	t=0
	rtbh_dump dump active distinct $GEOIPNOTTEXTLINE $GEOIPTEXTLINE format raw $TIME_TXT $LIMIT_TXT  mask /32 |\
	eval ${catchupSORT} | while read INPUT; do
		if [[ "$INPUT" == "" ]]; then break; fi
		if [[ "$LIMIT" != "" ]] && ((i==LIMIT)); then
			break
		fi
		if [[ "$catchupBITS" != "" ]]; then
			#our input is in decimal
			nCIDRA=($INPUT)	
			[[ -n "${CIDRA[0]}" ]] && v4dectest ${nCIDRA[0]}/32 ${CIDRA[0]}/$catchupBITS && continue
			CIDRA=($nCIDRA)
			INPUT="`inet_ntoa $INPUT`"
		fi
		rtbh_db_geoip_add $RTBH_ROUTE $INPUT && ((t++))
		if [[ "$TOTAL" != "" ]] && [[ "$t" -ge "$TOTAL" ]]; then break; fi
	done
}

rtbh_db_geoip_update()
{

	SELECT_TXT="select distinct seq,route from rtbh"
	WHERE_TXT="where stamp > datetime(CURRENT_TIMESTAMP,\"-$LONGEST_KEEP\")"
	if [[ "$1" != "all" ]]; then
		WHERE_TXT="$WHERE_TXT and (geoip_cc is NULL or geoip_cc = \"\") and active = 1"
	else
		shift
	fi
	if [[ "${1,,}" == "limit" ]]; then
		if [[ "$2" != "" ]]; then
			LIMIT=`printf %d $2`
			shift
		else
			LIMIT=100
		fi
		shift
	fi
	COUNT=(`echo "select count(all) from ($SELECT_TXT $WHERE_TXT);" | $SQLITE_CMD`) || return $?
	i=0	
	while OUTPUT=`echo "$SELECT_TXT $WHERE_TXT order by stamp DESC LIMIT 1 offset $i;" | $SQLITE_CMD`; do
		if [[ "$OUTPUT" == "" ]]; then break; fi
		if [[ "$LIMIT" != "" ]] && ((i==LIMIT)); then
			break
		fi
		sqlout_to_aar AOUTPUT OUTPUT || return $?
		NETCC=`rtbh_geoip_lookup_cc ${AOUTPUT['route']}`
		if [[ "${NETCC}" == "" ]]; then continue; fi
		((i++))
		UPDATE_TXT="update rtbh set geoip_cc = \"$NETCC\" $WHERE_TXT and seq = \"${AOUTPUT[seq]}\" ;"
		echo "$UPDATE_TXT" | $SQLITE_CMD || return $?
	done
	echo "$i of ${COUNT[2]} geoip_cc updated"
	return 0
}

rtbh_db_geoip()
{

	rtbh_geoip_enabled || return 1
	shift
	while [[ "$1" != "" ]]; do
	case "$1" in 
		catchup)
			if [[ "$2" != "" ]]; then
				if [[ -x "$2" ]] || [[ "`which $2 2>/dev/null`" != "" ]]; then
					RTBH_ROUTE="$2"
					shift
					shift
					rtbh_geoip_catchup $*
					return 0
				fi
			fi
			echo "This command mode expects the route add cmd as its first argument, not $2"
			;;
		update)
			shift 
			rtbh_db_geoip_update $*
			exit $?
			;;
		top)
			shift
			rtbh_db_geoip_top $*
			exit $?
			;;
		child)
			shift
			GEOIP_CHILD="$1"
			if [[ "`v4dec $1`" == "" ]]; then
				echo "This mode modifier expects a subnet, followed by the add or report modes"
				exit 1
			fi
			;;
		check)
			shift
			if [[ "$1" == "" ]]; then
				echo "This mode expect a subnet"
				exit 1
			fi
			rtbh_db_geoip_check $1
			return $?
			;;
		report)
			rtbh_db_geoip_report $*
			exit $?
			;;
		add)
			if [[ "$2" != "" ]]; then
				if [[ -x "$2" ]] || [[ "`which $2 2>/dev/null`" != "" ]]; then
					RTBH_ROUTE="$2"
					shift
					rtbh_db_geoip_add $*
					return 0
				fi
			fi
			echo "This command mode expects the route add cmd as its first argument"
			;;
		*)
			echo "This command mode accepts report, child, add, top, update, catchup arguments."
			exit 1
			;;
	esac
		shift
	done
	return 0
}

case "$db_mode" in


	list | listall)

		rtbh_db_list $*
		exit $?
		;;

	check)
		rtbh_db_check $*
		exit $?
		;;
	add)
		rtbh_db_add $*
		exit $?
		;;
	geoip)
		rtbh_db_geoip ${argv[*]}
		;;
	refresh)
		rtbh_db_refresh $*
		exit $?
		;;
	del | delete | rem | remove | forcedel) 
		rtbh_db_del $*
		exit $?
		;;
	update)
		rtbh_db_update_exp
		exit $?
		;;
	clean)
		rtbh_db_update_exp || exit $?
		if [[ "$argc" -ge 2 ]] && [[ -x "`which ${argv[1]}`" ]]; then
			rtbh_db_clean "${argv[1]}"
		else
			echo "This command mode expects as arguments the check-route cmd" 
			exit 1
		fi
		;;

	delexp)

		rtbh_db_update_exp || exit $?
		if [[ "$argc" -ge 2 ]] && [[ -x "`which ${argv[1]}`" ]]; then
			rtbh_db_delete_exp "${argv[1]}"
		else
			echo "This command mode expects as arguments the delete-route cmd" 
			exit 1
		fi
		;;


	route*sync |  route*synconly)

		rtbh_db_update_exp || exit $?
		if [[ "$argc" -ge 4 ]] && [[ -x "`which ${argv[1]}`" ]] &&\
		   [[ -x "`which ${argv[2]}`" ]] && [[ -x "`which ${argv[3]}`" ]]; then
			rtbh_db_route_sync "${argv[1]}" "${argv[2]}" "${argv[3]}"
			argv[2]="${argv[2]}"	
			argv[1]="${argv[3]}"	
			if [[ "$argc" -ge 5 ]]; then
				argv[3]="${argv[4]}"	
			fi
			if [[ "$argv[3]" == "stop" ]] || [[ "${db_mode##*sync}" == "only" ]]; then
				exit 0
			fi
			if [[ "${db_mode##route}" == "fsync" ]] || [[ "${db_mode##routes}" == "fsync" ]]; then
				FULL_SYNC="yes"
			fi
			((argc--))
		else
			echo "This command mode expects as arguments the route-show, add and delete cmds in that order"
			exit 1
		fi
		;&

	fsync)
		if [[ "$db_mode" == "fsync" ]]; then
			FULL_SYNC="yes"
		fi

		;&

	sync | syncroute?)
		rtbh_db_update_exp || exit $?
		if [[ "$argc" -ge 3 ]] && [[ -x "`which ${argv[1]}`" ]] && [[ -x "`which ${argv[2]}`" ]]; then
			if [[ "$argc" -ge 4 ]] && [[ "${argv[3]}" != "" ]]; then
				if [[ "${argv[3]}" == "add" ]]; then
					rtbh_db_sync "${argv[2]}" 1 
				elif [[ "${argv[3]}" == "del" ]]; then
					rtbh_db_sync "${argv[1]}" 0 
				else
					echo "Third argument to this command is either add or del"
					exit 1
				fi
			else
				FIRST="$((RANDOM%2))"
				SECOND="$((FIRST==0))"
				rtbh_db_sync "${argv[$((FIRST+1))]}" $FIRST
				rtbh_db_sync "${argv[$((SECOND+1))]}" $SECOND 
			fi
		else
			echo "This command mode expects as arguments the delete and add cmds, in that order"
			exit 1
		fi

		;;

	init)
		if [[ "$argc" == "2" ]] && [[ "${argv[1]}" == "init" ]]; then
                        logger -p $LOGFACPRI -t `basename $0` $USER initializing RTBH database
			echo "Initializing"
			which flock 2>&1 >/dev/null && flock $RTBH_DB -c "echo -n > $RTBH_DB"
			echo "drop table if exists rtbh;" | $SQLITE_CMD
			echo "$TABLE_CREATE" | $SQLITE_CMD
		else
			echo "Not initializing, use $0 init init to initiliaze which will DESTROY all current data"
			exit 1
		fi
		;;
	dump)
		rtbh_dump ${argv[*]} || exit $?
		;;
		
	*)
		echo "$0: supported command modes are check, add, geoip, del, forcedel, update, init, clean, delexp, sync, fsync, dump, list, listall, refresh"
		exit 1
		;;
esac
