/*
	llist A linked list and manipulation "library"

        Copyright 2002, Joe Maimon, New York USA


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "llist.h"

#ifdef _LLIST_ERROR_
#include "llist_error.h"
#endif /* _LLIST_ERROR_ */

struct linked_list *
add_linked_list_item(struct linked_list *list,char *data)
{
	list = push_linked_list_item(list,(void *)data,strlen(data)+1);	
	return(list);
}

struct linked_list *
push_linked_list_item(struct linked_list * list,void * data,size_t size)
{
	struct linked_list * newlist = NULL;

	
	if (!(size && data))
	{
#ifdef _LLIST_ERROR_
		LLIST_ERROR_THROW(EINVAL,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
		return(list);
	}
	
	newlist = (struct linked_list *) malloc ( sizeof(struct linked_list));
	if (!newlist)
	{
#ifdef _LLIST_ERROR_
		LLIST_ERROR_THROW(ENOMEM,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
		return(list);
	}

	newlist->data = NULL;
	newlist->next = NULL;
	newlist->prev = NULL;
	newlist->size = size;
	

	newlist->data = (void *) malloc ( newlist->size );
	if (!newlist->data)
	{
#ifdef _LLIST_ERROR_
		LLIST_ERROR_THROW(ENOMEM,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
		free(newlist);
		return(list);
	}
	
	memcpy(newlist->data,data,newlist->size);
	
	if (list)
	{
		list->next = newlist;
		newlist->prev = list;
	}
	
	return(newlist);
	
	

}

struct linked_list *
del_linked_list_item(struct linked_list *list)
{
	struct linked_list *newlist = NULL;
	struct linked_list *nlistp = NULL;
	
	if (!list)
	{
#ifdef _LLIST_ERROR_
		LLIST_ERROR_THROW(EINVAL,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
		return(NULL);
	}

	/* First the data */
	/* If you have use of the data, set list->data=NULL before calling this */
	if (list->data)
		free(list->data);
	
	if (list->next == NULL && list->prev != NULL)
	{
		/* We are at the end, more behind us and nothing in front */
		newlist = list->prev;
		newlist->next = NULL;
		free(list);
		return(newlist);
	}
	if (list->next != NULL && list->prev == NULL)
	{
		/* We are at the begining, more to come and nothing behind */
		newlist = list->next;
		newlist->prev = NULL;
		free(list);
		return(newlist);
	}
	if (list->next != NULL && list->prev != NULL)
	{
		/* We are in the middle of it now */
		/* We choose to advance the pointer 'forward' */
		newlist = list->next;
		nlistp = list->prev;
		nlistp->next = newlist;
		newlist->prev = nlistp;
		free(list);
		return(newlist);
	}
	if (list->next == NULL && list->prev == NULL)
	{
		/* Last remaining Item */
		free(list);
		return(NULL);
	}
	/* 
	* We should never get here. 
	* At this point we are looking at a big bug
	*/
#ifdef _LLIST_ERROR_
	LLIST_ERROR_THROW(LLIST_ERROR_EDELE,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
	return(list);
		
}
	
struct linked_list *
del_linked_list(struct linked_list *list)
{
	while (list)
		list = del_linked_list_item(list);
	
	return(list);
}

struct linked_list *
popdel_linked_list_item(struct linked_list * list,void ** data,size_t * len)
{
	if(!(data && list && list->data))
	{
#ifdef _LLIST_ERROR_
		LLIST_ERROR_THROW(EINVAL,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
		return(list);
	}
	list = pop_linked_list_item(list,data,len);
	list->data = NULL; /* So that the list forgets about this item */
	list = del_linked_list_item(list);
	return (list);
}

struct linked_list *
pop_linked_list_item(struct linked_list * list,void ** data,size_t * len)
{
	if(!(data && list && list->data))
	{
#ifdef _LLIST_ERROR_
		LLIST_ERROR_THROW(EINVAL,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
		return(list);
	}
	if(len)
		*len = list->size;
	*data = list->data;
	return(list);
	
}

struct linked_list *
getanddel_linked_list_item(struct linked_list * list,void ** data,size_t * len)
{
	return(popdel_linked_list_item(list,data,len));
			
}

struct linked_list *
popnext_linked_list_item(struct linked_list * list,void ** data,size_t * len)
{
	struct linked_list * nlistp = NULL; 
	list = pop_linked_list_item(list,data,len);
	if( (nlistp = next_linked_list_item(list)) )
		return(nlistp);
	else
	if( (nlistp = prev_linked_list_item(list)) )
		return(nlistp);
	else
		return(list);
	
}	

void *
linked_list_data(struct linked_list * list, size_t * len)
{
	if(!list)
		return(NULL);
	
	if(len)
		*len = list->size;
	return(list->data);
}	

char *
linked_list_item(struct linked_list * list)
{
	return((char *) linked_list_data(list,0));
}


struct linked_list *
next_linked_list_item(struct linked_list * list)
{
	list = step_linked_list(list,1);
	return(list);
}

struct linked_list *
prev_linked_list_item(struct linked_list * list)
{
	list = step_linked_list(list,0);
	return(list);
}

struct linked_list *
step_linked_list(struct linked_list * list,const int dir)
{
	/* 0 == FIFO, 0< == LIFO */	

	if (!list)
	{
#ifdef _LLIST_ERROR_
		LLIST_ERROR_THROW(EINVAL,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
		return(list);
	}

	if (!dir)
	{
		list = list->prev;
	}
	else
	{
		list = list->next;
	}
	return(list);
	
}


struct linked_list *
orient_linked_list(struct linked_list * list,const int dir)
{
	if (!list)
	{
#ifdef _LLIST_ERROR_
		LLIST_ERROR_THROW(EINVAL,__FILE__,__LINE__,__func__);
#endif /* _LLIST_ERROR_ */
		return(list);
	}

	if (!dir)
	{
		while (list->prev)
			list = list->prev;	
	}
	else
	{
		while (list->next)
			list = list->next;
	}
	return(list);
}

struct linked_list *
rewind_linked_list(struct linked_list * list)
{
	return(orient_linked_list(list,0));
}

struct linked_list *
ffrwrd_linked_list(struct linked_list * list)
{
	return(orient_linked_list(list,1));
}



ssize_t 
count_linked_list_data(struct linked_list * list)
{
	size_t size = 0;
	
	if(!list)
	{
		return((size_t)0);
	}

	list = orient_linked_list(list,0);

	if(!list)
		return((size_t)0);

	while(list)
	{
		size += list->size; 
		list = list->next;
	}
	return(size);
}	

unsigned int
count_linked_list_items(struct linked_list * list)
{
	int count = 0;

	if(!list)
	{
		return(0);
	}

	list = orient_linked_list(list,0);

	if(!list)
		return(0);

	while(list)
	{
		count++;
		list = next_linked_list_item(list);
	}
	return(count);
}
	
unsigned int
count_linked_list(struct linked_list * list,const int dir)
{
	int count = 0;

	if(!list)
	{
		return(0);
	}

	while(list)
	{
		count++;
		list = step_linked_list(list,dir);
	}
	return(count);
}



