/*
	llist.h, llist.c
	Routines to implement linked lists.
        
	Copyright 2002, Joe Maimon, New York USA


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifndef INCLUDED_LINK_LIST
	#define INCLUDED_LINK_LIST 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

/* A bi-directional list originaly designed for char strings
* Can theoreticaly be used for anything.
* 
* DO NOT define space for the structure, just a pointer.
* When handed a NULL for the first time add_linked_list_item || push_linked_list_item
* will allocate all neccessary and initialize.
* 
* If handed garbage, you will get a segfault. SO initialize your pointer to NULL.
* To use:
* struct linked_list * list = NULL;
*/

struct linked_list
{
        void  * data;
        struct linked_list *next;
        struct linked_list *prev;
	size_t size;
};


/* takes a list pointer and char string pointer, stores it and returns list address
* If the list does not currently exist, ensure the pointer *list == NULL and a new chain will be malloced.
* Returns *list (arg 1) on error (No mem)
* This is safe to use with primary list handle as it will not lose a valid list pointer.
* 
* TO detect whether value was stored correctly, compare return poiinter with input pointer. 
* If they match, no storage was done.
* 
* This is a wrapper arounf push_linked_list_item for char strings.
* TO Use:
* list = add_linked_list_item(list, string);
*/

struct linked_list *
add_linked_list_item(struct linked_list *list,char *data);

/*See add_linked_list_item()
 *
 * This is the general case of the above, it takes a (void *) to an opaque data structure.
 * You must supply the size.
 * 
 * TO Use:
 * list = push_linked_list_item(list,(void *)data,(int) data_size);
 */
struct linked_list *
push_linked_list_item(struct linked_list * list,void * data, size_t size);

/* takes list pointer. Free char string and list item. Direction is default to LIFO
* returns a new list pointer or NULL if the list is done for.
*
* This is safe to use on primary list handle because when you get NULL back, 
* there is nothing left to lose.
*/

struct linked_list *
del_linked_list_item(struct linked_list *list);

/* Takes list pointer. recursively calls del_linked_list_item until list is no more
* Returns NULL.
* 
* Use this on the primary list handle because it should be NULL after you call this.
* 
*/

struct linked_list *
del_linked_list(struct linked_list *list);

/* These two pop the data pointer and unlink/free the current list entry. 
 * There is NO mallocing here, the data pointer is copied.
 *
 * Refer to pop_linked_list_item and del_linked_list_item
 */

struct linked_list *
getanddel_linked_list_item(struct linked_list *list,void ** data,size_t * len);

struct linked_list *
popdel_linked_list_item(struct linked_list * list,void ** data,size_t * len);

/* Takes a list pointer, a pointer to a pointer of proper type, and a pointer to a size_t variable.
 * If data pointer is NULL nothing is stored there. If length pointer is NULL, length information is NOT stored there.
 * You need to make sure you can detect the end of the data refereneced by the pointer in that case.
 *
 * Does not delete the list item.
 * Does not move the list position.
 * Will not alter list value.
 *
 * You must use this in conjunction with {step|next|prev}_linked_list_item which may return a NULL
 * Test for NULL before assigning it your list handle because otherwise you risk a memory leak.
 *
 * Call count_linked_list_items to obtain a loop countdown.
 *
 * If there is an error data will remain NULL and so will len.
 */

struct linked_list *
pop_linked_list_item(struct linked_list * list,void ** data,size_t * len);

/* The only difference in this one is that the list posistion is moved.
 * If not possible (because the end of the list is here) it will return list verbatim. 
 * It is a good idea to orient the list while using this.
 * 
 * DO NOT test for NULL to terminate while loops while using this. Instead call count_linked_list_items.
 * No deleting
 */

struct linked_list *
popnext_linked_list_item(struct linked_list * list,void ** data,size_t * len);

/*
 * Returns a pointer to the linked lists data. Does not free.
 * Sets size value.
 */

void *
linked_list_data(struct linked_list * list, size_t * len);

/* Calls above but assumes char *
 */

char *
linked_list_item(struct linked_list * list);


/* Takes list pointer and moves forward */
struct linked_list *
next_linked_list_item(struct linked_list * list);

/* Takes list pointer and moves backwards */
struct linked_list *
prev_linked_list_item(struct linked_list * list);

/* Steps one link in any direction of list 
 * (DIR) ? LIFO : FIFO
 */
struct linked_list *
step_linked_list(struct linked_list * list,int dir);

/* Takes list pointer and flag for which direction to orient list
* (DIR) ? LIFO : FIFO
*/

struct linked_list *
orient_linked_list(struct linked_list *list,const int dir);

/*
 * Convenience functions for above
 */
struct linked_list *
rewind_linked_list(struct linked_list *list);

struct linked_list *
ffrwrd_linked_list(struct linked_list *list);

/*
* Takes pointer to list; returns total byte count of all data pointed to list->data_ptr
* On error will return -1
*/

ssize_t count_linked_list_data(struct linked_list * list);


/*
* Takes pointer to list; return count of ALL linked_list structures pointed to.
* 
* Return 0 on error (passing a null or bad list in)
*/

unsigned int 
count_linked_list_items(struct linked_list * list);

/*
* Takes pointer to list; return count of linked_list structures. 
* This will only count till the end of the list in the direction you specify.
* Return 0 on error (passing a null or bad list in)
*/

unsigned int 
count_linked_list(struct linked_list * list,const int dir);

#endif /* INCLUDED_LINK_LIST */
